//********************************************************************************************
//                                                                                           *
// AB&T Tecnologie Informatiche - Ivrea Italy                                                *
// http://www.bausano.net                                                                    *
// https://www.ethercat.org/en/products/791FFAA126AD43859920EA64384AD4FD.htm                 *
//                                                                                           *  
//********************************************************************************************    
//
//
//********************************************************************************************    
//                                                                                           *
// This software is distributed as an example, "AS IS", in the hope that it could            *
// be useful, WITHOUT ANY WARRANTY of any kind, express or implied, included, but            *
// not limited,  to the warranties of merchantability, fitness for a particular              *
// purpose, and non infringiment. In no event shall the authors be liable for any            *    
// claim, damages or other liability, arising from, or in connection with this software.     *
//                                                                                           *
//******************************************************************************************** 



//---- AB&T EasyCAT Gateway ------------------------------------------------------------------  


#define VerNumber 0x10                            // firmware version number  V_1.0                                            

#include "EasyCAT.h"                              // EasyCAT library for LAN9252
#include <Ethernet.h>                             // Ethernet library for wiz5500
#include <SPI.h>                                  // SPI library
#include <Wire.h>                                 // I2C library

#define EEPROM_ADD  0x50                          // EEPROM I2C address
#define BLINK_SPEED 1000                          // led blink time 1S


#define END         0xC0                          // constants for the Slip protocol
#define ESC         0xDB                          //      "
#define ESC_END     0xDC                          //      "
#define ESC_ESC     0xDD                          //      "


EasyCAT EASYCAT;                                  // EasyCAT istantiation

EthernetServer server(80);                        // Ethernet server instantiation on port 80 (HTTP default port)                      



//---- pins declaration ------------------------------------------------------------------------------


const int Res_Eth = 38;                           // wiz5500 reset pin
const int Led = 13;                               // led


//---- global variables ---------------------------------------------------------------------------

uint8_t Mac[6];                   

uint8_t EcatState;

uint8_t NumRxByte;
uint8_t PreviousRxByte;

uint8_t RxBuff[TOT_BYTE_NUM_IN];

uint8_t SlipTxBuff[TOT_BYTE_NUM_OUT * 2];
uint8_t WrIdx; 

bool IpAssigned;

uint32_t Time;
uint32_t UsbTime;
uint32_t BlinkTime;


//---- setup ---------------------------------------------------------------------------------------
 
void setup()
{
  uint8_t i;

  NumRxByte = 0;  
  IpAssigned = false;  

  Serial.begin(115200);                                 // init the debug serial line
  
  Wire.begin();                                         // init the I2C interface
  
  pinMode (Led, OUTPUT);                                // init the pins
  pinMode (Res_Eth, OUTPUT);                            //
 
  digitalWrite (Res_Eth, LOW);                          // two flashes on the led
  digitalWrite (Led, LOW);                              // and reset the wiz5500
  delay(300);                                           //
  digitalWrite (Res_Eth,HIGH);                          //
  digitalWrite (Led, HIGH);                             //
  delay(300);                                           //
  digitalWrite (Led, LOW);                              //
  delay(300);                                           //
  digitalWrite (Led, HIGH);                             //    
  delay(1000);                                          //
  digitalWrite (Led, LOW);                              // 
                                                                  
  Serial.println("\nEasyCAT Gateway basic example");    // print the banner
  Serial.print("Firmware version V_");                  //
  Serial.print(VerNumber >> 4, HEX);                    // extract Major and Minor from the version number
  Serial.print(".");                                    //
  Serial.println(VerNumber & 0x0F, HEX);                //    
  Serial.println("");                                   //
    

                                                        //------ initialize the EtherCAT interface ------ 
                                                        
  if (EASYCAT.Init() == true)                           // initialization
  {                                                     // successfully completed
    Serial.println("EtherCAT interface initialized\n"); //
  }                                                     //
  
  else                                                  // initialization failed   
  {                                                     // the EasyCAT interface was not recognized
    Serial.print ("initialization failed");             //     
                                                        // stay in loop for ever
    while(1)                                            // with the led blinking
    {                                                   //   
      digitalWrite (Led, HIGH);                         // 
      delay(200);                                       //   
      digitalWrite (Led, LOW);                          //  
      delay(200);                                       // 
    }                                                   // 
  }                                                     //

                                                        //------ initialize the Ethernet interface ------ 
                                                      
  Wire.beginTransmission(EEPROM_ADD);                   // read the MAC address from the EEPROM
  Wire.write(0xFA);                                     //
  Wire.endTransmission(0);                              //
  Wire.requestFrom(EEPROM_ADD,6,1);                     //

  Serial.print ("Ethernet interface Mac Address\n");    // print it out and store it
                                                        //
  i = 0;      
                                           
  while(Wire.available())                               //
  {                                                     //  
    Mac[i] = Wire.read();                               //
    Serial.print(Mac[i], HEX);                          //
    i++;                                                //
    if (i < 6)                                          //  
    {                                                   //
      Serial.print(":");                                //
    }                                                   //
  }                                                     //   
  Serial.println("");                                   //


  if (Ethernet.linkStatus() == LinkOFF)                 // nobody is connected to                 
  {                                                     // our Ethernet interface
    Serial.println("Ethernet cable is disconnected");   //
  }                                                     //
                                                            
  else if (Ethernet.linkStatus() == LinkON)             // somebody is connected to 
  {                                                     // our Ethernet interface  
    Serial.println("Ethernet cable is connected");      //
                                                        //
    Ethernet.begin(Mac);                                // obtain an IP address from the DHCP    
    Serial.println("Ethernet interface IP address");    // print it out
    Serial.println(Ethernet.localIP());                 //
    IpAssigned = true;                                  // 
                                                        //
    server.begin();                                     // start the web server
    Serial.print("Web server started\n");               //
  }                                                     //


  UsbTime = millis();
  BlinkTime = millis();  
}



//---- main loop ----------------------------------------------------------------------------------------
 
void loop()                                        
{               
  EcatState = EASYCAT.MainTask();                                   //------ execute the EtherCAT task --------------     


                                                                    //------ communication with the USB port --------
                                                                    //
  if (CheckRx())                                                    // if a valid frame has been received from the USB host
  {                                                                 //
    memcpy(&EASYCAT.BufferIn.Byte[0], &RxBuff[0], TOT_BYTE_NUM_IN); // copy the data received to the EtherCAT interface
                                                                    //
    TxOutBuff();                                                    // answer with the data from the EtherCAT interface
    UsbTime = millis();                                             // refresh the timer 
  }      


                                                                    //------ simple WEB server example ---------------                                                        
  if (IpAssigned)                                     
  {   
    EthernetClient client = server.available();                     // listen for incoming clients
    if (client) 
    {
      //Serial.println("new client");                               // debug     
      bool currentLineIsBlank = true;                               // an http request ends with a blank line
      
      while (client.connected()) 
      {
        if (client.available()) 
        {
          char c = client.read();
          //Serial.write(c);                                        // debug
               
          if (c == '\n' && currentLineIsBlank)                      // if we receive a a newline character
          {                                                         // and the line is blank, the http request 
                                                                    // has ended, so we can send a reply
           
            client.println("HTTP/1.1 200 OK");                      // send a standard http response header
            client.println("Content-Type: text/html");
            client.println("Connection: close");                    // the connection will be closed after completion of the response
            client.println("Refresh: 2");                           // refresh the page automatically every 2 sec
            client.println();
            client.println("<!DOCTYPE HTML>");
            client.println("<html>");
  
       
            client.println("<H1><font face=\"courier\" color=\"red\">");
            client.print("Easy");    
            client.print("<font color=\"black\">");    
            client.print("CAT Gateway test webpage</H1><br/>");
            client.println("<hr />"); 
  
            client.println("<H2>");         
  
            for (int i = 0; i<16; i++) 
            {                
              client.print("Byte_out");                             // visualize the outputs from 0 to 15
              client.print(i);                                      //
              client.print("&nbsp;&nbsp;");                         //   
                                                                    //
              if (i<10)                                             //
              {                                                     //
                client.print("&nbsp;");                             //
              }                                                     //
                                                                    //
              client.print("0x");                                   //                 
              if (EASYCAT.BufferOut.Byte[i] < 0x10)                 //   
              {                                                     //
                client.print("0");                                  //
              }                                                     //
              client.print(EASYCAT.BufferOut.Byte[i], HEX);         //    
  
              client.print("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;");
  
              client.print("Byte_out");                             // visualize the outputs from 16 to 31
              client.print(i + 16);                                 //
              client.print("&nbsp;&nbsp;");                         //
                                                                    //
              client.print("0x");                                   //
              if (EASYCAT.BufferOut.Byte[i + 16] < 0x10)            //
              {                                                     //
                client.print("0");                                  //
              }                                                     //
              client.print(EASYCAT.BufferOut.Byte[i + 16], HEX);    //
                                                                                  
              client.println("<br />");                                           
            }                                                                       
  
            client.println("</html>");
            break;
          }
          
          if (c == '\n')                                            // a new line is started
          {
            currentLineIsBlank = true;
          }
          
          else if (c != '\r')                                       // character received on the current line
          {
            currentLineIsBlank = false;
          }
        }
      }
      
      delay(1);                                                     // give the web browser time to receive the data
  
      client.stop();                                                // close the connection:
      //Serial.println("client disconnected");                      // debug
    }
  } 


  Time = millis();                                                  //------ led blink management -------------- 
                                                                    //
  if ((Time - UsbTime) > 500)                                       // if there is no communication with 
  {                                                                 // the USB host for more then 500 mS
      digitalWrite (Led, LOW);                                      // switch off the led
  }                                                                 //
  else                                                              // otherwise blink  
  {                                                                 // it at 1 S rate
    if ((Time - BlinkTime) >  BLINK_SPEED)                          //
    {                                                               //  
      BlinkTime = Time;                                             //
      digitalWrite (Led, (!digitalRead(Led)));                      //
    }                                                               //
  }                                                                 //
  
}


//---- functions ----------------------------------------------------------------------------------------



//---- slip frame transmission to the host --------------------------------------------------------------
                                                        //
                                                        // We transmit the output data to the host, encapsulated in a slip frame
                                                        // We use a variant in which the frame begins as well ends with an END character
                                                        // 
void TxOutBuff(void)

{                                                                                          
    uint8_t i;                                        
                                                        //----- The Slip frame is created in the SlipTxBuff buffer  ---------------  
                                                        //    
                                                        //
    WrIdx = 0;                                          // reset write index
                                                        // 
    SlipTxBuff[WrIdx++] = END;                          // add the Slip delimiter (start frame)
                                                        //
    for (i=0; i<TOT_BYTE_NUM_OUT; i++)                  // loop to add the data from the EtherCAT interface
    {                                                   // 
      AddByteToSlipFrame (EASYCAT.BufferOut.Byte[i]);   //   
    }                                                   //    
                                                        //
    AddByteToSlipFrame(EcatState);                      // add the EtherCAT state
    AddByteToSlipFrame(VerNumber);                      // add the firmware version
    SlipTxBuff[WrIdx++] = END;                          // add the Slip delimiter (end frame)
                                                        //
    SerialUSB.write(SlipTxBuff, WrIdx);                 // send the frame to the host through the native USB interface 
}


//---- add a byte to the Slip frame ---------------------------------------------------------------------

void AddByteToSlipFrame (uint8_t Data)
{

  switch (Data)          
  {
    case END:
      SlipTxBuff[WrIdx++] = ESC;
      SlipTxBuff[WrIdx++] = ESC_END;
    break;

    case ESC:
      SlipTxBuff[WrIdx++] = ESC ;
      SlipTxBuff[WrIdx++] = ESC_ESC;
    break;   

    default:
     SlipTxBuff[WrIdx++] = Data;
    break;
  }
}


//---- frame reception from the host --------------------------------------------------------------------                                           
                                                        //
                                                        // We analyze the data received from the host                         
                                                        // and unpack the payload of the Slip frame
                                                        //
                                                        // If a complete frame has been received, "true" is returned                                                                                                                                  
bool CheckRx(void)                  
                                                        
{
  uint8_t RxByte;
  bool FrameReady;

  FrameReady = false;
 
  while (SerialUSB.available())                         // Check if something is arrived from the host
  {                                                     // through the native USB interface
    RxByte = SerialUSB.read();                          //
    {                                                   //
      switch(RxByte)                                    // analize the byte received
      { 
        case ESC:                                       //--- it is ESC: we wait for the next byte ------
          PreviousRxByte = RxByte;                      // remember it
        break;                                          //
                    
        case END:                                       //--- it is END: may be the frame start or end  -                
          PreviousRxByte = RxByte;                      // remember it 
                                                        // 
          if (NumRxByte == TOT_BYTE_NUM_IN)             // If we have already received all the bytes expected     
          {                                             // it is a frame end and so the frame is valid                 
            FrameReady = true;                          // notify the main that a valid frame has been received 
          }                                             //
          NumRxByte = 0;                                // anyway reset the counter of received bytes   
        break;                                          //
                    
        default:                                        //--- it is a normal byte -----------------------
                                                        //    
          if(PreviousRxByte == ESC)                     // if the previous received byte was ESC 
          {                                             // we must decode it 
            PreviousRxByte = RxByte;                    // remember it
                                                        //
            switch(RxByte)                              // analyze it   
            {                                           //        
              case ESC_END:                             // it is ESC_END 
                RxByte = END;                           // so we have received END
              break;                                    //
                                                        //
              case ESC_ESC:                             // it is ESC_ESC
                RxByte = ESC;                           // so we have received ESC   
              break;                                    //    
            }                                           //
          }                                             //    
                                                        //
          else                                          // otherwise we use it as it is
          {                                             //           
            PreviousRxByte = RxByte;                    //
          }                                             //-----------------------------------------------
            
          RxBuff[NumRxByte] = RxByte;                   // store the decoded byte into the buffer                
          NumRxByte++;                                  // increment the counter of received bytes
                                                        // 
          if(NumRxByte > TOT_BYTE_NUM_IN)               // If we have received more bytes than expected
          {                                             // reset the counter of received bytes   
            NumRxByte = 0;                              //
          }                                             //    
        break;                                          //                        
      }                                                 //
    }                                                   //
  }                                                     //
  return FrameReady;                                    //
}                                                       //




